/*******************************************************************************
  march.c:  This is a nicely parameterizable march-test.  It returns boolean
  fail indication (i.e., true for pass, false for fail).  0 (false) is used for 
  pass so that each type of failure can be indicated other values.  See march.h
  for more details.  
  
  The activity for each pattern is separated out into a separate "marchPass()" 
  function because of limits in tester pattern RAM size.  "march()" itself 
  returns a composite 32-bit value:  The high-order 16 bits are the pattern 
  number at which a failure (if any) occured, and the low-order 16 bits is
  the failure type (see #defines in march.h).
  
  This will work for the KRAM and the cache's data array, once it is carefully
  loaded in sequence and then locked down.  That would needs to be done in 
  ASM(), to prevent it from accessing any memory while it loads the cache.
*******************************************************************************/
#include "m5282evb.h"
#include "environment.h"
#include "march.h"

int marchPass (
   UINT32 *baseAddr,
      /* base address of the memory array.  Must be a multiple of 4!!*/
   UINT32 size,  /* size of the array in bytes.  Must be a multiple of 4!!*/
   UINT32 pattern  /* the longword pattern to test with on this pass*/
) {
   UINT32 *addr;  /* pointer to the longword we're currently "dealing with"*/
   UINT32 *temp;
   UINT32 temp2;
   

   COMMENT ("First flood-fill the memory with the pattern, - step 1\n");
   for (addr = baseAddr;  addr < baseAddr + (size / 4); *(addr++) = pattern) {}
   COMMENT ("One by one, read/check, write complement, readback/check, - step 2:\n");
   for (addr = baseAddr;  addr < baseAddr + (size / 4);  addr++) {
      if (*addr != pattern)
         return FAILFILL;  /* fail*/
      *addr = ~pattern;
      if (*addr != ~pattern)
         return FAILCOMPL1;  /* fail*/
   }
   COMMENT ("One by one, read/check, invert it back, readback/check, - step 3:\n");
   for (addr = baseAddr;  addr < baseAddr + (size / 4);  addr++) {
      if (*addr != ~pattern)
         return FAILCOMPL2;  /* fail*/
      *addr = pattern;
      if (*addr != pattern)
         return FAILPAT2;
   }
   COMMENT ("One by one downward, read/check, write complement, readback/check, - step 4:\n");
   for (addr = baseAddr + (size / 4) - 1;  addr >= baseAddr;  addr--) {
      if (*addr != pattern)
         return FAILPAT3;  /* fail*/
      *addr = ~pattern;
      if (*addr != ~pattern)
         return FAILCOMPL3;
   }
   COMMENT ("One by one downward, read/check, invert it back, readback/check, - step 5:\n");
   for (addr = baseAddr + (size / 4) - 1;  addr >= baseAddr;  addr--) {
      if (*addr != ~pattern)
         return FAILCOMPL4;
      *addr = pattern;
      if (*addr != pattern)
         return FAILPAT4;
   }
   COMMENT ("One by one downward, read/check - step 6:\n");
   for (addr = baseAddr + (size / 4) - 1;  addr >= baseAddr;  addr--)
      if (*addr != pattern)
         return FAILPAT5;
   return MARCHPASS;  /* if we make it this far, we passed the test.*/
}


UINT32 pat[5] = {0x55555555, 0x33333333, 0x0f0f0f0f, 0x00ff00ff, 0x0000ffff};


int march (
   UINT32 *baseAddr,
      /* base address of the memory array.  Must be a multiple of 4!!*/
   UINT32 size  /* size of the array in bytes.  Must be a multiple of 4!!*/
) {
   int nPat;  /* the pattern we're working with;  see pat[] above*/
   int patResult;  /* return result from running a pattern*/
   
   for (nPat = 0;  nPat < 5;  nPat++) {
      patResult = marchPass (baseAddr, size, pat[nPat]);
      if (patResult != MARCHPASS)
         return (nPat << 16) | (patResult);
   }
   return MARCHPASS;  /* if we make it this far, we passed the test.*/
}
